﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace MicroRWD.MFIC
{
    // Encapsulates the status byte
    public class Status
    {
        #region Public Properties

        // b0 Internal EEPROM write error
        public bool EEPROM_Error { get; set; }
       
        // b1 Card serial number matched to identity code list
        public bool CardOK { get; set; }
        
        // b2 Card/Label communication and acknowledgement OK
        public bool RxOK { get; set; }
        
        // b3 Host serial communication error
        public bool RS232_Error { get; set; }
        
        // b4 MF type (0 = MF 1k byte card, 1 = MF 4k byte card)
        // (ICODE ONLY)
        public bool MF_Type { get; set; }
        
        // b5 UL type (0 = MF standard 1k/4k card, SINGLE UID), 1 = MF Ultralight card, DOUBLE UID)
        // (ICODE ONLY)
        public bool UL_Type { get; set; }
        
        // b6 MFRC error (Internal or antenna fault)
        public bool MFRC_Error { get; set; }

        // Validity
        public bool Valid { get; set; }

        #endregion


        #region Constructors

        // Construct an empty invalid Status instance
        public Status()
        {
            SetValue(0);
        }

        // Construct an initialised Status instance
        public Status(byte status)
        {
            SetValue(status);
        }

        #endregion


        #region Public Methods

        // Get the byte value from the current state
        public byte GetValue()
        {
            byte result;

            if (Valid)
            {
                // All valid MicroRWD.MFIC responses have b7 set
                result = 0x80;               
                // b0 Internal EEPROM write error
                result |= (byte)(EEPROM_Error ? (1 << 0) : 0);
                // b1 Card serial number matched to identity code list
                result |= (byte)(CardOK ? (1 << 1) : 0);
                // b2 Card/Label communication and acknowledgement OK
                result |= (byte)(RxOK ? (1 << 2) : 0);
                // b3 Host serial communication error
                result |= (byte)(RS232_Error ? (1 << 3) : 0);
                // b4 MF type (0 = MF 1k byte card, 1 = MF 4k byte card)
                // (ICODE ONLY)
                result |= (byte)(MF_Type ? (1 << 4) : 0);
                // b5 UL type (0 = MF standard 1k/4k card, SINGLE UID), 1 = MF Ultralight card, DOUBLE UID)
                // (ICODE ONLY)
                result |= (byte)(UL_Type ? (1 << 5) : 0);
                // b6 MFRC error (Internal or antenna fault)
                result |= (byte)(MFRC_Error ? (1 << 6) : 0);
            }
            else
            {
                // All bits zero indicates lack of a response
                result = 0x00;
            }

            // Return result;
            return result;
        }

        // Set the current state based on a byte value
        public void SetValue(byte _status)
        {
            // All valid MicroRWD responses have b7 set
            if ((_status & 0x80) == 0x80)
            {
                // b0 Internal EEPROM write error
                EEPROM_Error = (_status & (1 << 0)) != 0;
                // b1 Card serial number matched to identity code list
                CardOK = (_status & (1 << 1)) != 0;
                // b2 Card/Label communication and acknowledgement OK
                RxOK = (_status & (1 << 2)) != 0;
                // b3 Host serial communication error
                RS232_Error = (_status & (1 << 3)) != 0;
                // b4 MF type (0 = MF 1k byte card, 1 = MF 4k byte card)
                // (ICODE ONLY)
                MF_Type = (_status & (1 << 4)) != 0;
                // b5 UL type (0 = MF standard 1k/4k card, SINGLE UID), 1 = MF Ultralight card, DOUBLE UID)
                // (ICODE ONLY)
                UL_Type = (_status & (1 << 5)) != 0;
                // b6 MFRC error (Internal or antenna fault)
                MFRC_Error = (_status & (1 << 6)) != 0;
                // Valid
                Valid = true;
            }
            else
            {
                // b0 Internal EEPROM write error
                EEPROM_Error = false;
                // b1 Card serial number matched to identity code list
                CardOK = false;
                // b2 Card/Label communication and acknowledgement OK
                RxOK = false;
                // b3 Host serial communication error
                RS232_Error = false;
                // b4 MF type (0 = MF 1k byte card, 1 = MF 4k byte card)
                // (ICODE ONLY)
                MF_Type = false;
                // b5 UL type (0 = MF standard 1k/4k card, SINGLE UID), 1 = MF Ultralight card, DOUBLE UID)
                // (ICODE ONLY)
                UL_Type = false;
                // b6 MFRC error (Internal or antenna fault)
                MFRC_Error = false;
                // Valid
                Valid = false;
            }
        }

        // Express the Status as a bit string
        public override string ToString()
        {
            // Somewhere to hold the result
            string result;

            // Convert to bits
            
            // b7 All valid MicroRWD responses have b7 set
            //    All bits zero indicates lack of a response
            result = Valid ? "1 " : "0 ";
            // b6 MFRC error (Internal or antenna fault)
            result += MFRC_Error ? "1 " : "0 ";
            // b5 UL type (0 = MF standard 1k/4k card, SINGLE UID), 1 = MF Ultralight card, DOUBLE UID)
            // (ICODE ONLY)
            result += UL_Type ? "1 " : "0 ";
            // b4 MF type (0 = MF 1k byte card, 1 = MF 4k byte card)
            // (ICODE ONLY)
            result += MF_Type ? "1 " : "0 ";
            // b3 Host serial communication error
            result += RS232_Error ? "1 " : "0 ";
            // b2 Card/Label communication and acknowledgement OK
            result += RxOK ? "1 " : "0 ";
            // b1 Card serial number matched to identity code list
            result += CardOK ? "1 " : "0 ";
            // b0 Internal EEPROM write error
            result += EEPROM_Error ? "1 " : "0 ";

            // Return result
            return result;
        }

        #endregion
    }
}
